﻿using Microsoft.Web.Http;
using Microsoft.Xrm.Client.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using PpmsDataWebService.Models;
using PpmsDataWebService.Mappers;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Threading.Tasks;
using System.Web;
using System.Web.Http;
using System.Web.Http.Description;
using PpmsDataWebService.ModelsEnumTypes;
using VA.PPMS.Context;
using Microsoft.AspNet.OData;
using Microsoft.AspNet.OData.Routing;
using Microsoft.Xrm.Sdk.Messages;
using PpmsDataWebService.Helpers;
using Microsoft.OData;
using Microsoft.AspNet.OData.Extensions;


namespace PpmsDataWebService.Controllers
{
    //[Authorize]
    [EnableQuery]
    [ApiVersion("1.0")]
    public class ProvidersController : ODataController
    {

        [ODataRoute("Providers({ProviderIdentifier})")]
        public async Task<HttpResponseMessage> Get([FromODataUri] long ProviderIdentifier)
        //public async Task<SingleResult<Provider>> GetProvider([FromODataUri] int ProviderIdentifier)
        {

            //We will use the Query type found in URL to determine if Expanded properties need to be mapped. 
            var queryType = HttpContext.Current.Request.Url.Query;

            //This Scenario matches a Provider on it's Identifier. 
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                string providerIdentifier = ProviderIdentifier.ToString();
                var ppmsProvider = context.AccountSet.FirstOrDefault(i => i.ppms_ProviderIdentifier.Equals(providerIdentifier) && i.StatusCode.Value == (int)Account_StatusCode.Active);
                if (ppmsProvider != null)
                {

                    var ppmsProvidersList = new List<Account> { ppmsProvider };
                    var provider =
                        await ProviderMap.MapProviders(ppmsProvidersList, queryType);
                    //return SingleResult.Create(provider);
                    return Request.CreateResponse(provider);
                }
            }

            var message = string.Format("An Active Provider with Identifier: {0} not found", ProviderIdentifier);
            HttpError err = new HttpError(message);
            return Request.CreateErrorResponse(HttpStatusCode.OK, err);
        }

        [ODataRoute("Providers")]
        public async Task<HttpResponseMessage> Get()
        {

            var queryType = HttpContext.Current.Request.Url.Query;

            //Dynamics Web API Query
            /*
            using (var client = new HttpClient())
            {
                client.BaseAddress = new Uri("https://dvagov-ppms-dev.crm9.dynamics.com/api/data/v8.2/");
                client.Timeout = new TimeSpan(0, 2, 0);
                var token = await PpmsContextHelper.GetToken();
                client.DefaultRequestHeaders.Authorization = new AuthenticationHeaderValue("Bearer", token);
                client.DefaultRequestHeaders.Add("OData-MaxVersion", "4.0");
                client.DefaultRequestHeaders.Add("OData-Version", "4.0");
                client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/json"));
                var result = await client.GetAsync("accounts" + queryType);
                var stringResult = await result.Content.ReadAsStringAsync();
                var providers = JsonConvert.DeserializeObject<dynamic>(stringResult);
                
                return Request.CreateResponse(stringResult);
            }
            

            //Odata URI Parser. Not using for now but could be useful later. 
            //var serviceRootUri = new Uri(HttpContext.Current.Request.Url.AbsoluteUri);
            //var pathAndQueryUri = new Uri(HttpContext.Current.Request.Url.AbsoluteUri);
            //var model = WebApiConfig.GetEdmModel();
            //ODataUriParser parser = new ODataUriParser(model, serviceRootUri, pathAndQueryUri);
            //FilterClause filter = parser.ParameterAliasNodes();
            */

            //This Scenario returns the first 10 Providers in the system. . 

            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                var ppmsProviders = context.AccountSet.Where(i => i.ppms_ProviderIdentifier != null && i.StatusCode.Value == (int)Account_StatusCode.Active).Take(10);

                var ppmsProvidersList = ppmsProviders.ToList();
                if (ppmsProvidersList.Any())
                {
                    //Map Providers
                    var providers = await ProviderMap.MapProviders(ppmsProvidersList, "None");
                    return Request.CreateResponse(providers);
                }
                var message = string.Format("No Providers were found");
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }



        //[HttpGet]
        //[MapToApiVersion("1.0")]
        //[ResponseType(typeof(Provider))]
        [ODataRoute("GetProviderByName")]
        public async Task<HttpResponseMessage> GetProviderByName([FromODataUri] string name)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                //Find Top 500 Providers matching name given
                var ppmsProviders = context.AccountSet.Where(i => i.Name.Contains(name) && i.StatusCode.Value == (int)Account_StatusCode.Active).Take(500);
                var ppmsProvidersList = ppmsProviders.ToList();
                if (ppmsProvidersList.Any())
                {
                    //Map Providers
                    var providers = await ProviderMap.MapProviders(ppmsProvidersList, "None");
                    return Request.CreateResponse(providers);
                }
                var message = string.Format("No Providers found matching: " + name);
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }


        [ODataRoute("GetProvidersByModifiedOnDate")]
        public async Task<HttpResponseMessage> GetProvidersByModifiedOnDate([FromODataUri] DateTime modifiedOnOrAfterDate, [FromODataUri] int networkId, [FromODataUri] int isExternal, [FromODataUri] int page)
        {
            var queryType = HttpContext.Current.Request.Url.Query;

            var _service = await PpmsContextHelper.GetProxy();

            ppms_vaprovidernetwork providerNetwork = null;
            if (networkId != 0)
            {
                using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
                {
                    providerNetwork = context.ppms_vaprovidernetworkSet.FirstOrDefault(i => i.ppms_networknumber == networkId && i.StatusCode.Value == (int)ppms_vaprovidernetwork_StatusCode.Active);
                }
                if (providerNetwork == null)
                {
                    var noNetworkMessage = string.Format("No Network Found based on Network Id given");
                    HttpError noNetworkErr = new HttpError(noNetworkMessage);
                    return Request.CreateErrorResponse(HttpStatusCode.NotFound, noNetworkErr);
                }
            }

            //Format the Modified On Date
            var modifiedOnOrAfterDateString = modifiedOnOrAfterDate.ToString("yyyy-MM-dd");

            QueryExpression qe = new QueryExpression();
            qe.EntityName = "account";
            qe.ColumnSet.AllColumns = true;
            qe.Distinct = true;

            //Initialize Filter Expression for Provider
            var filterExpression = new FilterExpression();
            filterExpression.FilterOperator = LogicalOperator.And;
            var condition1 = new ConditionExpression("statuscode", ConditionOperator.Equal, ((int)Account_StatusCode.Active).ToString());
            filterExpression.AddCondition(condition1);
            var condition2 = new ConditionExpression("modifiedon", ConditionOperator.GreaterEqual, modifiedOnOrAfterDateString);
            filterExpression.AddCondition(condition2);
            //Check if IsExternal is needed
            if (isExternal == 0 || isExternal == 1)
            {
                var condition3 = new ConditionExpression("ppms_isexternal", ConditionOperator.Equal, isExternal.ToString());
                filterExpression.AddCondition(condition3);
            }

            qe.Criteria.AddFilter(filterExpression);

            if (networkId != 0 && providerNetwork != null)
            {

                var linkedEntity = new LinkEntity
                {
                    JoinOperator = JoinOperator.Inner,
                    LinkFromAttributeName = "accountid",
                    LinkFromEntityName = Account.EntityLogicalName,
                    LinkToAttributeName = "ppms_providerid",
                    LinkToEntityName = ppms_providerservice.EntityLogicalName,

                    LinkCriteria =
                        {
                            Filters =
                            {
                                new FilterExpression
                                {
                                    FilterOperator = LogicalOperator.And,
                                    Conditions =
                                    {
                                        new ConditionExpression("ppms_network", ConditionOperator.Equal, providerNetwork.Id),
                                        new ConditionExpression("statuscode", ConditionOperator.Equal, ((int)ppms_providerservice_StatusCode.Active).ToString())
                                    },
                                },
                            }
                        }
                };
                qe.LinkEntities.Add(linkedEntity);
            }

            // Query using the paging cookie.
            // Define the paging attributes.
            // The number of records per page to retrieve.
            int queryCount = 5000;
            // Initialize the page number.
            int pageNumber = 1;
            // Assign the pageinfo properties to the query expression.
            qe.PageInfo = new PagingInfo();
            qe.PageInfo.Count = queryCount;
            qe.PageInfo.PageNumber = pageNumber;
            // The current paging cookie. When retrieving the first page, 
            // pagingCookie should be null.
            qe.PageInfo.PagingCookie = null;

            //Initialize the List of Providers
            List<Account> allProvidersList = new List<Account>();

            while (true)
            {
                EntityCollection ec = _service.RetrieveMultiple(qe);

                if (ec.Entities != null)
                {
                    //Add to the Provider list
                    allProvidersList.AddRange(ec.Entities.Select(e => e.ToEntity<Account>()).ToList());
                }

                if (ec.MoreRecords)
                {
                    qe.PageInfo.PageNumber++;

                    // Set the paging cookie to the paging cookie returned from current results.
                    qe.PageInfo.PagingCookie = ec.PagingCookie;
                }
                else
                {
                    // If no more records are in the result nodes, exit the loop.
                    break;
                }
            }

            var resultsPerPage = 10000;
            var list = splitProviders(allProvidersList, resultsPerPage).ToList();

            if (allProvidersList.Count() == 0)
            {
                var noProvidersmessage = string.Format("No Modified Providers Found");
                HttpError noProvidersErr = new HttpError(noProvidersmessage);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, noProvidersErr);
            }
            //Make Sure the page cannot be out of range.
            if (page > list.Count())
            {
                var maxReachedmessage = string.Format("No Additional pages of Providers");
                HttpError maxErr = new HttpError(maxReachedmessage);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, maxErr);
            }

            List<Account> mapProvidersList = new List<Account>();

            //Prevent Page from being out of range. 
            //Default to the first page if page is 0 or lower 
            if (page <= 0)
            {
                page = 1;
            }

            mapProvidersList = list[page - 1];



            if (mapProvidersList.Any())
            {
                var metadataList = new List<Response> { };
                var metadata = new Response { };

                //Verify we want to Expand & Map Providers
                if ((queryType.Contains("expand") || queryType.Contains("Expand")) && (queryType.Contains("providers") || queryType.Contains("Providers")))
                {
                    //Custom Provider Map  
                    var providers = await ProviderMap.MapProviders(mapProvidersList, queryType);
                    metadata.Providers = providers;
                }
                metadata.Status = "Success";
                metadata.ReturnedResults = mapProvidersList.Count();
                metadata.TotalResults = allProvidersList.Count();
                metadata.MaxResultsPerPage = resultsPerPage;
                metadata.PageNumber = page;
                metadata.TotalPages = list.Count();
                metadata.Timestamp = DateTime.UtcNow;
                metadataList.Add(metadata);
                var response = metadataList.AsQueryable();
                return Request.CreateResponse(response);
            }
            var message = string.Format("No Providers found matching criteria");
            HttpError err = new HttpError(message);
            return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
        }

        
        [EnableQuery]
        [ODataRoute("GetModifiedProviders")]
        public async Task<HttpResponseMessage> GetModifiedProviders([FromODataUri] int networkId, [FromODataUri] int isExternal, [FromODataUri] int page)
        {
            var queryType = HttpContext.Current.Request.Url.Query;

            var _service = await PpmsContextHelper.GetProxy();

            ppms_vaprovidernetwork providerNetwork = null;
            if (networkId != 0)
            {
                using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
                {
                    providerNetwork = context.ppms_vaprovidernetworkSet.FirstOrDefault(i => i.ppms_networknumber == networkId && i.StatusCode.Value == (int)ppms_vaprovidernetwork_StatusCode.Active);
                }
                if (providerNetwork == null)
                {
                    var noNetworkMessage = string.Format("No Network Found based on Network Id given");
                    HttpError noNetworkErr = new HttpError(noNetworkMessage);
                    return Request.CreateErrorResponse(HttpStatusCode.NotFound, noNetworkErr);
                }
            }

            //Only return providers that were modified in the last day. 
            var modifiedOnOrAfterDateString = DateTime.Now.AddHours(-24).ToString("yyyy-MM-dd");

            QueryExpression qe = new QueryExpression();
            qe.EntityName = "account";
            qe.ColumnSet = new ColumnSet("accountid", "ppms_provideridentifier", "statuscode");
            qe.Distinct = true;
            //qe.ColumnSet.AllColumns = true;

            //Initialize Filter Expression for Provider
            var filterExpression = new FilterExpression();
            filterExpression.FilterOperator = LogicalOperator.And;
            var condition1 = new ConditionExpression("modifiedon", ConditionOperator.GreaterEqual, modifiedOnOrAfterDateString);
            filterExpression.AddCondition(condition1);
            //Check if IsExternal is needed
            if (isExternal == 0 || isExternal == 1)
            {
                var condition2 = new ConditionExpression("ppms_isexternal", ConditionOperator.Equal, isExternal.ToString());
                filterExpression.AddCondition(condition2);
            }

            qe.Criteria.AddFilter(filterExpression);

            if (networkId != 0 && providerNetwork != null)
            {

                var linkedEntity = new LinkEntity
                {
                    JoinOperator = JoinOperator.Inner,
                    LinkFromAttributeName = "accountid",
                    LinkFromEntityName = Account.EntityLogicalName,
                    LinkToAttributeName = "ppms_providerid",
                    LinkToEntityName = ppms_providerservice.EntityLogicalName,


                    LinkCriteria =
                        {
                            Filters =
                            {
                                new FilterExpression
                                {
                                    FilterOperator = LogicalOperator.And,
                                    Conditions =
                                    {
                                        new ConditionExpression("ppms_network", ConditionOperator.Equal, providerNetwork.Id),
                                        new ConditionExpression("statuscode", ConditionOperator.Equal, ((int)ppms_providerservice_StatusCode.Active).ToString())
                                    },
                                },
                            }
                        }
                };
                qe.LinkEntities.Add(linkedEntity);
            }

            // Query using the paging cookie.
            // Define the paging attributes.
            // The number of records per page to retrieve.
            int queryCount = 5000;
            // Initialize the page number.
            int pageNumber = 1;
            // Assign the pageinfo properties to the query expression.
            qe.PageInfo = new PagingInfo();
            qe.PageInfo.Count = queryCount;
            qe.PageInfo.PageNumber = pageNumber;
            // The current paging cookie. When retrieving the first page, 
            // pagingCookie should be null.
            qe.PageInfo.PagingCookie = null;

            //Initialize the List of Providers
            List<Account> allProvidersList = new List<Account>();

            while (true)
            {
                EntityCollection ec = _service.RetrieveMultiple(qe);

                if (ec.Entities != null)
                {
                    //Add to the Provider list
                    allProvidersList.AddRange(ec.Entities.Select(e => e.ToEntity<Account>()).ToList());
                }

                if (ec.MoreRecords)
                {
                    qe.PageInfo.PageNumber++;

                    // Set the paging cookie to the paging cookie returned from current results.
                    qe.PageInfo.PagingCookie = ec.PagingCookie;
                }
                else
                {
                    // If no more records are in the result nodes, exit the loop.
                    break;
                }
            }

            var resultsPerPage = 25000;
            var list = splitProviders(allProvidersList, resultsPerPage).ToList();

            if (allProvidersList.Count() == 0)
            {
                var noProvidersmessage = string.Format("No Modified Providers Found");
                HttpError noProvidersErr = new HttpError(noProvidersmessage);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, noProvidersErr);
            }
            //Make Sure the page cannot be out of range.
            if (page > list.Count())
            {
                var maxReachedmessage = string.Format("No Additional pages of Providers");
                HttpError maxErr = new HttpError(maxReachedmessage);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, maxErr);
            }

            List<Account> mapProvidersList = new List<Account>();

            //Prevent Page from being out of range. 
            //Default to the first page if page is 0 or lower 
            if (page <= 0)
            {
                page = 1;
            }

            mapProvidersList = list[page - 1];

            if (mapProvidersList.Any())
            {

                var metadataList = new List<Response> { };
                var metadata = new Response { };

                //Verify we want to Expand & Map Providers
                if ((queryType.Contains("expand") || queryType.Contains("Expand")) && (queryType.Contains("providers") || queryType.Contains("Providers")))
                {
                    //Custom Provider Map  
                    var providers = await QuickProviderMap.MapProviders(mapProvidersList, queryType);
                    metadata.Providers = providers;
                }
                metadata.Status = "Success";
                metadata.ReturnedResults = mapProvidersList.Count();
                metadata.TotalResults = allProvidersList.Count();
                metadata.MaxResultsPerPage = resultsPerPage;
                metadata.PageNumber = page;
                metadata.TotalPages = list.Count();
                metadata.Timestamp = DateTime.UtcNow;
                metadataList.Add(metadata);
                var response = metadataList.AsQueryable();
                return Request.CreateResponse(response);
            }
            var message = string.Format("No Modified Providers Found");
            HttpError err = new HttpError(message);
            return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);

        }

        public static IEnumerable<List<Account>> splitProviders<Account>(List<Account> providers, int nSize = 30)
        {
            for (int i = 0; i < providers.Count; i += nSize)
            {
                yield return providers.GetRange(i, Math.Min(nSize, providers.Count - i));
            }
        }



        [ODataRoute("GetProviderNetworks")]
        public async Task<HttpResponseMessage> GetProviderNetworks([FromODataUri] long npi)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                string providerIdentifier = npi.ToString();
                var ppmsProvider = context.AccountSet.FirstOrDefault(i => i.ppms_ProviderIdentifier.Equals(providerIdentifier) && i.StatusCode.Equals((int)Account_StatusCode.Active));
                if (ppmsProvider != null)
                {
                    //Look Up Provider Services first
                    var ppmsProviderServices = context.ppms_providerserviceSet.Where(i => i.ppms_ProviderId.Id == ppmsProvider.Id && i.StatusCode.Value == (int)ppms_providerservice_StatusCode.Active);
                    var ppmsProviderServicesList = ppmsProviderServices.ToList();

                    if (ppmsProviderServicesList.Any())
                    {
                        QueryExpression qe = new QueryExpression();
                        qe.EntityName = "ppms_vaprovidernetwork";
                        qe.ColumnSet = new ColumnSet();
                        qe.ColumnSet.AllColumns = true;
                        qe.Criteria = new FilterExpression(LogicalOperator.Or);

                        foreach (var ppmsProviderService in ppmsProviderServicesList)
                        {
                            //Check Provider Network field
                            if (ppmsProviderService.ppms_network != null)
                            {
                                var filter = new FilterExpression();
                                filter.AddCondition("ppms_vaprovidernetworkid", ConditionOperator.Equal, new object[] { ppmsProviderService.ppms_network.Id });
                                filter.AddCondition("statuscode", ConditionOperator.Equal, ((int)ppms_vaprovidernetwork_StatusCode.Active).ToString());
                                qe.Criteria.AddFilter(filter);
                            }
                        }
                        EntityCollection ec = context.RetrieveMultiple(qe);

                        var ppmsProvidersNetworksList = new List<ppms_vaprovidernetwork>();

                        foreach (ppms_vaprovidernetwork providerNetwork in ec.Entities)
                        {
                            ppmsProvidersNetworksList.Add(providerNetwork);
                        }

                        if (ppmsProvidersNetworksList.Any())
                        {
                            var providerNetworks =
                                await ProviderNetworkMap.MapProviderNetworks(ppmsProvidersNetworksList, "none", context);
                            return Request.CreateResponse(providerNetworks);
                        }
                        else
                        {
                            var networkMessage = string.Format("No Active Provider Networks found for this Provider.");
                            HttpError networkeErr = new HttpError(networkMessage);
                            return Request.CreateErrorResponse(HttpStatusCode.NotFound, networkeErr);
                        }
                    }
                }
                var message = string.Format("No Provider found based on NPI given.");
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);


            }
        }

        [ODataRoute("GetProvidersByNetwork")]
        public async Task<HttpResponseMessage> GetProvidersByNetwork([FromODataUri] int networkId)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                //var network = NetworkIds.GetNetwork(networkId);
                var providerNetwork = context.ppms_vaprovidernetworkSet.FirstOrDefault(i => i.ppms_networknumber == networkId);
                if (providerNetwork != null)
                {
                    QueryExpression qe = new QueryExpression();
                    qe.EntityName = "account";
                    qe.ColumnSet = new ColumnSet();
                    qe.ColumnSet.AllColumns = true;

                    //Initialize Filter Expression for Provider
                    var filterExpression = new FilterExpression();
                    filterExpression.FilterOperator = LogicalOperator.And;
                    var condition = new ConditionExpression("statuscode", ConditionOperator.Equal, ((int)Account_StatusCode.Active).ToString());
                    filterExpression.AddCondition(condition);
                    qe.Criteria.AddFilter(filterExpression);

                    var linkedEntity = new LinkEntity
                    {
                        JoinOperator = JoinOperator.Inner,
                        LinkFromAttributeName = "accountid",
                        LinkFromEntityName = Account.EntityLogicalName,
                        LinkToAttributeName = "ppms_providerid",
                        LinkToEntityName = ppms_providerservice.EntityLogicalName,

                        LinkCriteria =
                        {
                            Filters =
                            {
                                new FilterExpression
                                {
                                    FilterOperator = LogicalOperator.And,
                                    Conditions =
                                    {
                                        new ConditionExpression("ppms_network", ConditionOperator.Equal, providerNetwork.Id)
                                    },
                                },
                            }
                        }
                    };
                    qe.LinkEntities.Add(linkedEntity);

                    //Initialzie Provider List to add to
                    List<Account> providerList = new List<Account>();
                    // Query using the paging cookie.
                    // Define the paging attributes.
                    // The number of records per page to retrieve.
                    int queryCount = 5000;
                    // Initialize the page number.
                    int pageNumber = 1;
                    // Assign the pageinfo properties to the query expression.
                    qe.PageInfo = new PagingInfo();
                    qe.PageInfo.Count = queryCount;
                    qe.PageInfo.PageNumber = pageNumber;
                    // The current paging cookie. When retrieving the first page, 
                    // pagingCookie should be null.
                    qe.PageInfo.PagingCookie = null;

                    while (true)
                    {

                        EntityCollection ec = context.RetrieveMultiple(qe);

                        if (ec.Entities != null)
                        {
                            //Add to the Provider list
                            providerList.AddRange(ec.Entities.Select(e => e.ToEntity<Account>()).ToList());
                        }

                        if (ec.MoreRecords)
                        {
                            qe.PageInfo.PageNumber++;

                            // Set the paging cookie to the paging cookie returned from current results.
                            qe.PageInfo.PagingCookie = ec.PagingCookie;
                        }
                        else
                        {
                            // If no more records are in the result nodes, exit the loop.
                            break;
                        }

                    }

                    //Map Providers
                    var providers =
                            await ProviderMap.MapProviders(providerList, "none");
                    return Request.CreateResponse(providers);

                }

                var message = string.Format("No Providers found for this network");
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }

        [ODataRoute("GetProvidersByAffiliationType")]
        public async Task<HttpResponseMessage> GetProvidersByAffiliationType([FromODataUri] int affiliationTypeId)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {

                //Look Up Provider Va Provider Relationship first
                var vaProviderRelationship = RelationshipTypeIds.GetRelationshipType(affiliationTypeId);

                var relationships = context.ppms_vaproviderrelationshipSet.Where(i => i.ppms_relationship.Equals(vaProviderRelationship.Value));
                var providerRelationships = relationships.ToList();

                if (providerRelationships.Count > 0)
                {
                    QueryExpression qe = new QueryExpression();
                    qe.EntityName = "account";
                    qe.ColumnSet = new ColumnSet();
                    qe.ColumnSet.AllColumns = true;

                    //Initialize Filter Expression for Provider
                    var filterExpression = new FilterExpression();
                    filterExpression.FilterOperator = LogicalOperator.And;
                    var condition = new ConditionExpression("statuscode", ConditionOperator.Equal, ((int)Account_StatusCode.Active).ToString());
                    filterExpression.AddCondition(condition);
                    qe.Criteria.AddFilter(filterExpression);

                    //Initialize Link Entity
                    var linkedEntity = new LinkEntity();
                    linkedEntity.JoinOperator = JoinOperator.Inner;
                    linkedEntity.LinkFromAttributeName = "accountid";
                    linkedEntity.LinkFromEntityName = Account.EntityLogicalName;
                    linkedEntity.LinkToAttributeName = "ppms_providerid";
                    linkedEntity.LinkToEntityName = ppms_providerservice.EntityLogicalName;

                    //Initialize Filter Expression for Link Entity
                    var linkFilterExpressionOr = new FilterExpression();
                    linkFilterExpressionOr.FilterOperator = LogicalOperator.Or;

                    //Add Condtions to the link Filter Expression 
                    foreach (var providerRelationship in providerRelationships)
                    {
                        var linkCondition = new ConditionExpression("ppms_vaproviderrelationshipid", ConditionOperator.Equal, providerRelationship.Id);
                        linkFilterExpressionOr.AddCondition(linkCondition);
                    };
                    //Add the filter expression to Link Criteria
                    linkedEntity.LinkCriteria.AddFilter(linkFilterExpressionOr);

                    //Linked Entity must be in Active Status
                    var linkFilterExpressionAnd = new FilterExpression();
                    linkFilterExpressionAnd.FilterOperator = LogicalOperator.And;
                    var linkConditionStatus = new ConditionExpression("statuscode", ConditionOperator.Equal, ppms_providerservice_StatusCode.Active);
                    linkFilterExpressionAnd.AddCondition(linkConditionStatus);
                    linkedEntity.LinkCriteria.AddFilter(linkFilterExpressionAnd);

                    //Add the Linked Entity to Query Expression
                    qe.LinkEntities.Add(linkedEntity);

                    //Initialzie Provider List to add to
                    List<Account> providerList = new List<Account>();
                    // Query using the paging cookie.
                    // Define the paging attributes.
                    // The number of records per page to retrieve.
                    int queryCount = 5000;
                    // Initialize the page number.
                    int pageNumber = 1;
                    // Assign the pageinfo properties to the query expression.
                    qe.PageInfo = new PagingInfo();
                    qe.PageInfo.Count = queryCount;
                    qe.PageInfo.PageNumber = pageNumber;
                    // The current paging cookie. When retrieving the first page, 
                    // pagingCookie should be null.
                    qe.PageInfo.PagingCookie = null;

                    while (true)
                    {

                        EntityCollection ec = context.RetrieveMultiple(qe);

                        if (ec.Entities != null)
                        {
                            //Add to the Provider list
                            providerList.AddRange(ec.Entities.Select(e => e.ToEntity<Account>()).ToList());
                        }

                        if (ec.MoreRecords)
                        {
                            qe.PageInfo.PageNumber++;

                            // Set the paging cookie to the paging cookie returned from current results.
                            qe.PageInfo.PagingCookie = ec.PagingCookie;
                        }
                        else
                        {
                            // If no more records are in the result nodes, exit the loop.
                            break;
                        }
                    }

                    //Map Providers
                    var providers =
                            await ProviderMap.MapProviders(providerList, "none");
                    return Request.CreateResponse(providers);

                }

                var message = string.Format("No Providers found for this Affiliation Type");
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }

        [ODataRoute("GetProvidersByRelationship")]
        public async Task<HttpResponseMessage> GetProvidersByRelationship([FromODataUri] int relationshipId)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {

                //Look Up Provider Va Provider Relationship first
                var vaProviderRelationship = VAProviderRelationshipIds.GetVAProviderRelationship(relationshipId);

                var providerRelationship = context.ppms_vaproviderrelationshipSet.FirstOrDefault(i => i.ppms_name.Equals(vaProviderRelationship.Name));

                if (providerRelationship != null)
                {
                    QueryExpression qe = new QueryExpression();
                    qe.EntityName = "account";
                    qe.ColumnSet = new ColumnSet();
                    qe.ColumnSet.AllColumns = true;

                    //Initialize Filter Expression for Provider
                    var filterExpression = new FilterExpression();
                    filterExpression.FilterOperator = LogicalOperator.And;
                    var condition = new ConditionExpression("statuscode", ConditionOperator.Equal, ((int)Account_StatusCode.Active).ToString());
                    filterExpression.AddCondition(condition);
                    qe.Criteria.AddFilter(filterExpression);

                    var linkedEntity = new LinkEntity
                    {
                        JoinOperator = JoinOperator.Inner,
                        LinkFromAttributeName = "accountid",
                        LinkFromEntityName = Account.EntityLogicalName,
                        LinkToAttributeName = "ppms_providerid",
                        LinkToEntityName = ppms_providerservice.EntityLogicalName,

                        LinkCriteria =
                        {
                            Filters =
                            {
                                new FilterExpression
                                {
                                    FilterOperator = LogicalOperator.And,
                                    Conditions =
                                    {
                                        new ConditionExpression("ppms_vaproviderrelationshipid", ConditionOperator.Equal, providerRelationship.Id)
                                    },
                                },
                            }
                        }
                    };
                    qe.LinkEntities.Add(linkedEntity);

                    //Initialzie Provider List to add to
                    List<Account> providerList = new List<Account>();
                    // Query using the paging cookie.
                    // Define the paging attributes.
                    // The number of records per page to retrieve.
                    int queryCount = 5000;
                    // Initialize the page number.
                    int pageNumber = 1;
                    // Assign the pageinfo properties to the query expression.
                    qe.PageInfo = new PagingInfo();
                    qe.PageInfo.Count = queryCount;
                    qe.PageInfo.PageNumber = pageNumber;
                    // The current paging cookie. When retrieving the first page, 
                    // pagingCookie should be null.
                    qe.PageInfo.PagingCookie = null;

                    while (true)
                    {

                        EntityCollection ec = context.RetrieveMultiple(qe);

                        if (ec.Entities != null)
                        {
                            //Add to the Provider list
                            providerList.AddRange(ec.Entities.Select(e => e.ToEntity<Account>()).ToList());
                        }

                        if (ec.MoreRecords)
                        {
                            qe.PageInfo.PageNumber++;

                            // Set the paging cookie to the paging cookie returned from current results.
                            qe.PageInfo.PagingCookie = ec.PagingCookie;
                        }
                        else
                        {
                            // If no more records are in the result nodes, exit the loop.
                            break;
                        }
                    }

                    //Map Providers
                    var providers =
                            await ProviderMap.MapProviders(providerList, "none");
                    return Request.CreateResponse(providers);

                }

                var message = string.Format("No Providers found for this Relationship");
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }


        [ODataRoute("GetProvidersByActiveDateRange")]
        public async Task<HttpResponseMessage> GetProvidersByActiveDateRange([FromODataUri] DateTime startDate, [FromODataUri] DateTime endDate)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                //Find all the History Log Records of type Provider Status - Active, and then retrieve the Providers
                var activeStatusHistoryLogs = from hl in context.ppms_historylogSet
                                              where hl.ppms_providerid != null
                                              where hl.ppms_type.Value == (int)ppms_historylog_ppms_type.ProviderStatus
                                              where hl.ppms_providerstatus.Value == (int)ppms_historylog_ppms_providerstatus.Active
                                              where hl.CreatedOn >= startDate.Date
                                              where hl.CreatedOn <= endDate.Date
                                              //where hl.ppms_statusenddate != null || hl.ppms_statusenddate !=null && hl.ppms_statusenddate.Value.Date >= endDate.Date
                                              //where hl.ppms_statusenddate.Value.Date >= endDate.Date
                                              select hl;

                var activeStatusHistoryLogsList = activeStatusHistoryLogs.ToList();
                if (activeStatusHistoryLogsList.Any())
                {
                    //Max Criteria for Query Expression is 500
                    var activeStatusHistoryLogsListParsed = activeStatusHistoryLogsList.Select((x, i) => new { Index = i, Value = x }).GroupBy(x => x.Index / 500).Select(x => x.Select(v => v.Value).ToList()).ToList();
                    var ppmsProvidersList = new List<Account>();
                    foreach (var activeHistoryLogList in activeStatusHistoryLogsListParsed)
                    {
                        QueryExpression qe = new QueryExpression();
                        qe.EntityName = "account";
                        qe.ColumnSet = new ColumnSet();
                        qe.ColumnSet.AllColumns = true;
                        qe.Criteria = new FilterExpression(LogicalOperator.Or);
                        foreach (var activeHistoryLog in activeHistoryLogList)
                        {
                            //Check the Status End Dates. 
                            if (activeHistoryLog.ppms_statusenddate != null && endDate >
                                activeHistoryLog.ppms_statusenddate.Value.ToUniversalTime().Date) continue;

                            var filter = new FilterExpression();
                            filter.AddCondition("accountid", ConditionOperator.Equal, new object[] { activeHistoryLog.ppms_providerid.Id });
                            qe.Criteria.AddFilter(filter);
                        }
                        EntityCollection ec = context.RetrieveMultiple(qe);

                        foreach (Account provider in ec.Entities)
                        {
                            ppmsProvidersList.Add(provider);
                        }
                    }
                    var providers =
                        await ProviderMap.MapProviders(ppmsProvidersList, "none");
                    return Request.CreateResponse(providers);

                }
                var message = string.Format("No Active Providers found on given Date Range.");
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }

        [ODataRoute("GetProvidersAuthorized")]
        public async Task<HttpResponseMessage> GetProvidersAuthorized([FromODataUri] long providerIdentifier, [FromODataUri] DateTime date, [FromODataUri] int network)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                string identifierMatch = providerIdentifier.ToString();
                var ppmsProvider = context.AccountSet.FirstOrDefault(i => i.ppms_ProviderIdentifier.Equals(identifierMatch));
                if (ppmsProvider != null)
                {
                    //var providerNetwork = NetworkIds.GetNetwork(network);
                    var providerNetwork = context.ppms_vaprovidernetworkSet.FirstOrDefault(i => i.ppms_networknumber == network);

                    if (ppmsProvider != null && providerNetwork != null)
                    {
                        //Look up and convert Prov Identifier
                        int provId = Int32.Parse(ppmsProvider.ppms_ProviderIdentifier);

                        var providerAuthorization = new ProviderAuthorization
                        {
                            Network = providerNetwork.ppms_name,
                            ProviderIdentifier = provId,
                            OnLeie = false,
                            ActiveOnNetworkBetweenDatesGiven = false
                        };

                        var ppmsProviderServices = context.ppms_providerserviceSet.Where(i => i.ppms_ProviderId.Id == ppmsProvider.Id && i.ppms_network.Id == providerNetwork.Id);
                        var ppmsProviderServicesList = ppmsProviderServices.ToList();
                        var ppmsVaProviderNetworksList = new List<ppms_vaprovidernetwork>();
                        var ppmsVaProviderRelationshipsList = new List<ppms_vaproviderrelationship>();
                        if (ppmsProviderServicesList.Any())
                        {
                            QueryExpression qe = new QueryExpression();
                            qe.EntityName = "ppms_vaprovidernetwork";
                            qe.ColumnSet = new ColumnSet();
                            qe.ColumnSet.AllColumns = true;
                            qe.Criteria = new FilterExpression(LogicalOperator.Or);
                            foreach (var ppmsProviderService in ppmsProviderServicesList)
                            {
                                //Check for Relationship
                                if (ppmsProviderService.ppms_network == null) continue;

                                var filter = new FilterExpression();
                                filter.AddCondition("ppms_vaprovidernetworkid", ConditionOperator.Equal, new object[] { ppmsProviderService.ppms_network.Id });
                                qe.Criteria.AddFilter(filter);
                            }
                            EntityCollection ec = context.RetrieveMultiple(qe);

                            foreach (ppms_vaprovidernetwork vpn in ec.Entities)
                            {
                                ppmsVaProviderNetworksList.Add(vpn);
                            }

                            //Retrieve VA Provider Relationships
                            QueryExpression qe2 = new QueryExpression();
                            qe2.EntityName = "ppms_vaproviderrelationship";
                            qe2.ColumnSet = new ColumnSet();
                            qe2.ColumnSet.AllColumns = true;
                            qe2.Criteria = new FilterExpression(LogicalOperator.Or);
                            foreach (var ppmsProviderNetwork in ppmsVaProviderNetworksList)
                            {
                                //Check for Relationship
                                if (ppmsProviderNetwork.ppms_VAProviderRelationshipId == null) continue;

                                var filter = new FilterExpression();
                                filter.AddCondition("ppms_vaproviderrelationshipid", ConditionOperator.Equal, new object[] { ppmsProviderNetwork.ppms_VAProviderRelationshipId.Id });
                                qe2.Criteria.AddFilter(filter);
                            }
                            EntityCollection ec2 = context.RetrieveMultiple(qe2);

                            foreach (ppms_vaproviderrelationship vpr in ec2.Entities)
                            {
                                ppmsVaProviderRelationshipsList.Add(vpr);
                            }

                            if (ppmsVaProviderRelationshipsList.Any())
                            {
                                foreach (var relationship in ppmsVaProviderRelationshipsList)
                                {
                                    if (date < relationship.ppms_participationstartdate.Value.ToUniversalTime().Date) continue;
                                    if (relationship.ppms_participationenddate != null && date > relationship.ppms_participationenddate.Value.ToUniversalTime().Date) continue;
                                    providerAuthorization.ActiveOnNetworkBetweenDatesGiven = true;
                                }
                            }

                            //Check if there is a History Log where the Provider was in Leie Status between the dates given. 
                            var providerStatusHistoryLogs = from hl in context.ppms_historylogSet
                                                            where hl.ppms_providerid != null
                                                            where hl.ppms_providerid.Id == ppmsProvider.Id
                                                            where hl.ppms_type.Value == (int)ppms_historylog_ppms_type.ProviderStatus
                                                            where hl.ppms_providerstatus.Value == (int)ppms_historylog_ppms_providerstatus.LEIEExclusion
                                                            select hl;

                            var providerSatusHistoryLogsList = providerStatusHistoryLogs.ToList();
                            if (providerSatusHistoryLogsList.Any())
                            {
                                foreach (var providerStatusHisotryLog in providerSatusHistoryLogsList)
                                {
                                    if (date < providerStatusHisotryLog.CreatedOn.Value.ToUniversalTime().Date) continue;
                                    if (providerStatusHisotryLog.ppms_statusenddate != null && date > providerStatusHisotryLog.ppms_statusenddate.Value.ToUniversalTime().Date) continue;
                                    providerAuthorization.OnLeie = true;
                                }
                            }
                            return Request.CreateResponse(providerAuthorization);
                        }
                    }
                }
            }
            var message = string.Format("Unable able to locate Provider or Network based on Identifier & Network given.");
            HttpError err = new HttpError(message);
            return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
        }

        //Returns only the Provider's related Specialties
        [ODataRoute("Providers({ProviderIdentifier})/ProviderSpecialties")]
        public async Task<HttpResponseMessage> GetProviderSpecialties([FromODataUri] long ProviderIdentifier)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {

                string identifierMatch = ProviderIdentifier.ToString();
                var ppmsProvider = context.AccountSet.FirstOrDefault(i => i.ppms_ProviderIdentifier.Equals(identifierMatch) && i.StatusCode.Equals((int)Account_StatusCode.Active));
                if (ppmsProvider != null)
                {
                    var ppmsProviderSpecialties =
                        context.ppms_providertaxonomySet.Where(i => i.ppms_ProviderId.Id == ppmsProvider.Id && i.StatusCode.Equals((int)ppms_providertaxonomy_StatusCode.Active));

                    var ppmsProviderSpecialtiesList = ppmsProviderSpecialties.ToList();
                    //Map Provider Record. 

                    var providerSpecialties =
                        await ProviderSpecialtyMap.MapProviderSpecialties(ppmsProviderSpecialtiesList, "None", context);
                    return Request.CreateResponse(providerSpecialties);

                }
                var message = string.Format("No Provider Specialties found for provider with Identifier: " + ProviderIdentifier);
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }

        [ODataRoute("Providers({ProviderIdentifier})/ProviderServices")]
        public async Task<HttpResponseMessage> GetProviderServices([FromODataUri] long ProviderIdentifier)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {

                string identifierMatch = ProviderIdentifier.ToString();
                var ppmsProvider = context.AccountSet.FirstOrDefault(i => i.ppms_ProviderIdentifier.Equals(identifierMatch) && i.StatusCode.Equals((int)Account_StatusCode.Active));
                if (ppmsProvider != null)
                {

                    //Map Provider Services
                    var ppmsProviderServices =
                        context.ppms_providerserviceSet.Where(i => i.ppms_ProviderId.Id == ppmsProvider.Id && i.StatusCode.Value == (int)ppms_providerservice_StatusCode.Active);
                    var ppmsProviderServicesList = ppmsProviderServices.ToList();
                    var providerServices =
                        await ProviderServicesMap.MapProviderServices(ppmsProviderServicesList, "None", context);
                    return Request.CreateResponse(providerServices);

                }
                var message = string.Format("No Provider Services found for provider with Identifier" + ProviderIdentifier);
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }

        [ODataRoute("Providers({ProviderIdentifier})/ProviderContacts")]
        public async Task<HttpResponseMessage> GetProviderContacts([FromODataUri] long ProviderIdentifier)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {

                string identifierMatch = ProviderIdentifier.ToString();
                var ppmsProvider = context.AccountSet.FirstOrDefault(i => i.ppms_ProviderIdentifier.Equals(identifierMatch) && i.StatusCode.Equals((int)Account_StatusCode.Active));
                if (ppmsProvider != null)
                {
                    //Map Provider Services
                    //Lookup and Map the Provider Contacts
                    var ppmsProviderContacts =
                        context.ContactSet.Where(i => i.ParentCustomerId.Id == ppmsProvider.Id);
                    var ppmsProviderContactsList = ppmsProviderContacts.ToList();
                    var providerContacts =
                        await ProviderContactsMap.MapProviderContacts(ppmsProviderContactsList, "None", context);
                    return Request.CreateResponse(providerContacts);

                }
                var message = string.Format("No Provider Contacts found for provider with IDentifier: " + ProviderIdentifier);
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }

        [ODataRoute("Providers({ProviderIdentifier})/Providers")]
        public async Task<HttpResponseMessage> GetRelatedProviders([FromODataUri] long ProviderIdentifier)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {

                string identifierMatch = ProviderIdentifier.ToString();
                var ppmsProvider = context.AccountSet.FirstOrDefault(i => i.ppms_ProviderIdentifier.Equals(identifierMatch) && i.StatusCode.Equals((int)Account_StatusCode.Active));
                if (ppmsProvider != null)
                {
                    //Map Providers
                    var ppmsRelatedProviders =
                        context.AccountSet.Where(i => i.ParentAccountId.Id == ppmsProvider.Id);
                    var ppmsRelatedProvidersList = ppmsRelatedProviders.ToList();
                    var relatedProviders =
                        await ProviderMap.MapProviders(ppmsRelatedProvidersList, "None");
                    return Request.CreateResponse(relatedProviders);

                }
                var message = string.Format("No Provider Contacts found for provider with IDentifier: " + ProviderIdentifier);
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }

        [ODataRoute("Providers({ProviderIdentifier})/ProviderIdentifiers")]
        public async Task<HttpResponseMessage> GetProviderIdentifiers([FromODataUri] long ProviderIdentifier)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                //int npi = ProviderIdentifier;
                string providerIdentifierMatch = ProviderIdentifier.ToString();
                var ppmsProvider = context.AccountSet.FirstOrDefault(i => i.ppms_ProviderIdentifier.Equals(providerIdentifierMatch) && i.StatusCode.Equals((int)Account_StatusCode.Active));
                if (ppmsProvider != null)
                {
                    //Lookup and map the Provider NPi's
                    var ppmsProviderIdentifiers =
                        context.ppms_provideridentifierSet.Where(i => i.ppms_Provider.Id == ppmsProvider.Id && i.StatusCode.Equals((int)ppms_provideridentifier_StatusCode.Active));
                    var ppmsProviderIdentifierList = ppmsProviderIdentifiers.ToList();
                    var providerIds =
                        await ProviderIdentifiersMap.MapProviderIdentifiers(ppmsProviderIdentifierList, "None", context);
                    return Request.CreateResponse(providerIds);

                }
                var message = string.Format("No Provider Npi's found for provider with NPI: " + ProviderIdentifier);
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }


        [ODataRoute("Providers({ProviderIdentifier})/ProviderLicenses")]
        public async Task<HttpResponseMessage> GetProviderLicenses([FromODataUri] long ProviderIdentifier)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {

                string identifierMatch = ProviderIdentifier.ToString();
                var ppmsProvider = context.AccountSet.FirstOrDefault(i => i.ppms_ProviderIdentifier.Equals(identifierMatch) && i.StatusCode.Equals((int)Account_StatusCode.Active));
                if (ppmsProvider != null)
                {

                    //Lookup and Map the Provider Licenses
                    var ppmsProviderLicenses =
                        context.ppms_providerlicensureSet.Where(i => i.ppms_ProviderIdId.Id == ppmsProvider.Id);
                    var ppmsProviderLicensesList = ppmsProviderLicenses.ToList();
                    var providerLicenses =
                        await ProviderLicensesMap.MapProviderLicenses(ppmsProviderLicensesList, "None", context);
                    return Request.CreateResponse(providerLicenses);

                }
                var message = string.Format("No Provider Licenses found for provider with Identifier: " + ProviderIdentifier);
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }


        [ODataRoute("Providers({ProviderIdentifier})/ProviderCredentials")]
        public async Task<HttpResponseMessage> GetProviderCredentials([FromODataUri] long ProviderIdentifier)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {

                string identifierMatch = ProviderIdentifier.ToString();
                var ppmsProvider = context.AccountSet.FirstOrDefault(i => i.ppms_ProviderIdentifier.Equals(identifierMatch) && i.StatusCode.Equals((int)Account_StatusCode.Active));
                if (ppmsProvider != null)
                {

                    //Lookup and Map the Provider Credentials
                    var ppmsProviderCredentials =
                        context.ppms_providercredentialSet.Where(i => i.ppms_ProviderId.Id == ppmsProvider.Id);
                    var ppmsProviderCredentialsList = ppmsProviderCredentials.ToList();
                    var providerCredentials =
                        await ProviderCredentialsMap.MapProviderCredentials(ppmsProviderCredentialsList, "None", context);
                    return Request.CreateResponse(providerCredentials);

                }
                var message = string.Format("No Provider Credentials found for provider with Identifier: " + ProviderIdentifier);
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }

        [ODataRoute("Providers({ProviderIdentifier})/ProviderPrivileges")]
        public async Task<HttpResponseMessage> GetProviderPrivileges([FromODataUri] long ProviderIdentifier)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {

                string identifierMatch = ProviderIdentifier.ToString();
                var ppmsProvider = context.AccountSet.FirstOrDefault(i => i.ppms_ProviderIdentifier.Equals(identifierMatch) && i.StatusCode.Equals((int)Account_StatusCode.Active));
                if (ppmsProvider != null)
                {

                    //Lookup and Map the Provider Privileges
                    var ppmsProviderPrivileges =
                        context.ppms_providerprivilegeSet.Where(i => i.ppms_ProviderId.Id == ppmsProvider.Id);
                    var ppmsProviderPrivilegesList = ppmsProviderPrivileges.ToList();
                    var providerPrivileges =
                        await ProviderPrivilegesMap.MapProviderPrivileges(ppmsProviderPrivilegesList, "None", context);
                    return Request.CreateResponse(providerPrivileges);

                }
                var message = string.Format("No Provider Privileges found for provider with Identifier: " + ProviderIdentifier);
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }

        [ODataRoute("Providers({ProviderIdentifier})/DEAs")]
        public async Task<HttpResponseMessage> GetProviderDeas([FromODataUri] long ProviderIdentifier)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {

                string identifierMatch = ProviderIdentifier.ToString();
                var ppmsProvider = context.AccountSet.FirstOrDefault(i => i.ppms_ProviderIdentifier.Equals(identifierMatch) && i.StatusCode.Equals((int)Account_StatusCode.Active));
                if (ppmsProvider != null)
                {

                    var ppmsDeas = context.ppms_deascheduleprivilegeSet.Where(i => i.ppms_ProviderId.Id == ppmsProvider.Id);
                    var ppmsDeasList = ppmsDeas.ToList();
                    if (ppmsDeasList.Any())
                    {
                        //Map Deas
                        var deas = await DeaMap.MapDea(ppmsDeasList, "None", context);
                        return Request.CreateResponse(deas);
                    }
                }
                var message = string.Format("No Care Sites found for provider with Identifer: " + ProviderIdentifier);
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }

        [ODataRoute("Providers({ProviderIdentifier})/CareSites")]
        public async Task<HttpResponseMessage> GetProviderCareSites([FromODataUri] long ProviderIdentifier)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {

                string identifierMatch = ProviderIdentifier.ToString();
                var ppmsProvider = context.AccountSet.FirstOrDefault(i => i.ppms_ProviderIdentifier.Equals(identifierMatch) && i.StatusCode.Equals((int)Account_StatusCode.Active));
                if (ppmsProvider != null)
                {

                    var ppmsCareSites = context.ppms_caresiteSet.Where(i => i.ppms_organization.Id == ppmsProvider.Id);
                    var ppmsCareSitesList = ppmsCareSites.ToList();
                    if (ppmsCareSitesList.Any())
                    {
                        //Map Care Sites
                        var careSites = await CareSiteMap.MapCareSites(ppmsCareSitesList, "None", context);
                        return Request.CreateResponse(careSites);
                    }
                }
                var message = string.Format("No Care Sites found for provider with Identifer: " + ProviderIdentifier);
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }


        [ODataRoute("Providers({ProviderIdentifier})/VaProviderRelationships")]
        public async Task<HttpResponseMessage> GetVaProviderRelationships([FromODataUri] long ProviderIdentifier)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {

                string identifierMatch = ProviderIdentifier.ToString();
                var ppmsProvider = context.AccountSet.FirstOrDefault(i => i.ppms_ProviderIdentifier.Equals(identifierMatch) && i.StatusCode.Equals((int)Account_StatusCode.Active));
                if (ppmsProvider != null)
                {
                    var ppmsProviderServices = context.ppms_providerserviceSet.Where(i => i.ppms_ProviderId.Id == ppmsProvider.Id);
                    var ppmsProviderServicesList = ppmsProviderServices.ToList();
                    var ppmsVaProviderRelationshipsList = new List<ppms_vaproviderrelationship>();
                    if (ppmsProviderServicesList.Any())
                    {
                        QueryExpression qe = new QueryExpression();
                        qe.EntityName = "ppms_vaproviderrelationship";
                        qe.ColumnSet = new ColumnSet();
                        qe.ColumnSet.AllColumns = true;
                        qe.Criteria = new FilterExpression(LogicalOperator.Or);
                        foreach (var ppmsProviderService in ppmsProviderServicesList)
                        {
                            //Check for Relationship
                            if (ppmsProviderService.ppms_VAProviderRelationshipId == null) continue;

                            var filter = new FilterExpression();
                            filter.AddCondition("ppms_vaproviderrelationshipid", ConditionOperator.Equal, new object[] { ppmsProviderService.ppms_VAProviderRelationshipId.Id });
                            qe.Criteria.AddFilter(filter);
                        }
                        EntityCollection ec = context.RetrieveMultiple(qe);

                        foreach (ppms_vaproviderrelationship vpr in ec.Entities)
                        {
                            ppmsVaProviderRelationshipsList.Add(vpr);
                        }
                        //Map Va Provider Relationships
                        var vaProviderRelationships = await VaProviderRelationshipMap.MapVaProviderRelationships(ppmsVaProviderRelationshipsList, "None", context);
                        return Request.CreateResponse(vaProviderRelationships);
                    }
                }
                var message = string.Format("No Va Provider Relationships found for provider with Identifer: " + ProviderIdentifier);
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.OK, err);
            }
        }




        [ODataRoute("Providers({ProviderIdentifier})/OwnedCareSite")]
        public async Task<HttpResponseMessage> GetProviderOwnedCareSite([FromODataUri] long ProviderIdentifier)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {

                string identifierMatch = ProviderIdentifier.ToString();
                var ppmsProvider = context.AccountSet.FirstOrDefault(i => i.ppms_ProviderIdentifier.Equals(identifierMatch) && i.StatusCode.Equals((int)Account_StatusCode.Active));
                if (ppmsProvider != null)
                {

                    //Make sure we are mapping for an Organization type Provider. 
                    if (ppmsProvider?.ppms_providertype?.Value ==
                        (int)Account_ppms_providertype.GroupPracticeAgency)
                    {
                        var ppmsOwnedCareSite =
                            context.ppms_caresiteSet.FirstOrDefault(
                                i => i.Id == ppmsProvider.ppms_ownedcaresite.Id);
                        var ppmsOwnedCareSiteList = new List<ppms_caresite> { ppmsOwnedCareSite };
                        if (ppmsOwnedCareSiteList.Any())
                        {
                            //Map Care Sites
                            var ownedCareSite =
                                await CareSiteMap.MapCareSites(ppmsOwnedCareSiteList, "None", context);
                            return Request.CreateResponse(ownedCareSite);
                        }
                    }
                }
                var message = string.Format("No Owned Care Site found for provider with Identifier: " + ProviderIdentifier);
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }

    }
}